package com.fredericboisguerin.insa.listdemo.view;

import java.awt.*;

import javax.swing.*;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;

import com.fredericboisguerin.insa.listdemo.presenter.ListDemoView;

public class SwingListDemoView extends JPanel implements ListDemoView {

    private static final String HIRE_LABEL = "Hire";
    private static final String FIRE_LABEL = "Fire";

    private final ListDemoListener listener;

    private final JList<String> list;
    private final JButton fireButton;
    private final JTextField employeeName;
    private final JButton hireButton;
    private final DefaultListModel<String> listModel;

    public SwingListDemoView(ListDemoListener listener) {
        super(new BorderLayout());
        this.listener = listener;

        this.list = buildJList();
        this.listModel = new DefaultListModel<>();
        list.setModel(listModel);
        list.addListSelectionListener(e -> updateFireButtonState());
        JScrollPane listScrollPane = new JScrollPane(this.list);

        hireButton = new JButton(HIRE_LABEL);
        HireListener hireListener = new HireListener();
        hireButton.addActionListener(e -> onHireActionPerformed());
        hireButton.setEnabled(false);

        fireButton = new JButton(FIRE_LABEL);
        fireButton.addActionListener(e -> onFireActionPerformed());

        employeeName = new JTextField(10);
        employeeName.addActionListener(e -> onHireActionPerformed()); // Hold "Enter" button hits
        employeeName.getDocument().addDocumentListener(hireListener);

        JPanel buttonPane = buildJPanelWith(fireButton, employeeName, hireButton);

        add(listScrollPane, BorderLayout.CENTER);
        add(buttonPane, BorderLayout.PAGE_END);
    }

    @Override
    public void addEmployee(String employeeName) {
        listModel.addElement(employeeName);
    }

    @Override
    public void addEmployee(int selectedIndex, String name) {
        listModel.insertElementAt(name, selectedIndex);
        select(selectedIndex);
    }

    @Override
    public void removeEmployee(int index) {
        listModel.remove(index);
        updateFireButtonState();
        if (index >= 0) {
            select(index - 1);
        }
    }

    private void onFireActionPerformed() {
        int index = list.getSelectedIndex();
        listener.onEmployeeFired(index);
    }

    private void onHireActionPerformed() {
        listener.onEmployeeHired(employeeName.getText(), list.getSelectedIndex());
        employeeName.requestFocusInWindow();
        employeeName.setText("");
    }

    private void select(int index) {
        list.setSelectedIndex(index);
        list.ensureIndexIsVisible(index);
    }

    private void updateFireButtonState() {
        fireButton.setEnabled(list.getSelectedIndex() >= 0);
    }

    class HireListener implements DocumentListener {

        @Override
        public void insertUpdate(DocumentEvent e) {
            updateHireButtonState(e);
        }

        @Override
        public void removeUpdate(DocumentEvent e) {
            updateHireButtonState(e);
        }

        @Override
        public void changedUpdate(DocumentEvent e) {
            updateHireButtonState(e);
        }

        private void updateHireButtonState(DocumentEvent e) {
            boolean notEmptyText = e.getDocument().getLength() > 0;
            hireButton.setEnabled(notEmptyText);
        }
    }

    private static JPanel buildJPanelWith(JButton fireButton, JTextField employeeName, JButton hireButton) {
        JPanel buttonPane = new JPanel();
        buttonPane.setLayout(new BoxLayout(buttonPane, BoxLayout.LINE_AXIS));
        buttonPane.add(fireButton);
        buttonPane.add(Box.createHorizontalStrut(5));
        buttonPane.add(new JSeparator(SwingConstants.VERTICAL));
        buttonPane.add(Box.createHorizontalStrut(5));
        buttonPane.add(employeeName);
        buttonPane.add(hireButton);
        buttonPane.setBorder(BorderFactory.createEmptyBorder(5, 5, 5, 5));
        return buttonPane;
    }

    private static JList<String> buildJList() {
        JList<String> list = new JList<>();
        list.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
        list.setSelectedIndex(0);
        list.setVisibleRowCount(5);
        return list;
    }

}