#! /usr/bin/env python3

import chipwhisperer as cw 
import time 
import sys
import numpy as np
import argparse
import struct
import matplotlib.pyplot as plt
from workbench import WorkBench
import readline

# --------------------------------------------------------------------------------------- #
#                                 Glitch Sweep parameters                                 #
# --------------------------------------------------------------------------------------- #

upload_firmware = True
firmware        = "glitch_loop.hex"
width_min       = 1
width_max       = 46
width_step      = 1
offset_min      = -48
offset_max      =  48
offset_step     = 1
ext_offset      = 920

# --------------------------------------------------------------------------------------- #
#                                                                                         #
# --------------------------------------------------------------------------------------- #

class MyGlitchResults :
	def __init__(self):
		self.results={"normal": [], "success": [], "reset": []} 

	def add(self, group, value_tuple):
		"""
		group: "normal", "success", or "reset"
		value_tuple: (width, offset, tries)
		"""
		if group in self.results:
			self.results[group].append(value_tuple)
		else:
			raise ValueError(f"Unknown group: {group}")

	def get_results(self):
		"""Return stored results."""
		return self.results	

def main():
	local_config = dict()
	
	wb = WorkBench(local_config.get('workbench', dict()), False)
	
	if upload_firmware:
		wb.upload_firmware(firmware)
		wb.reset()
		print("Firmware uploaded successfully")
	
	my_results = MyGlitchResults()
	gc = cw.GlitchController(groups=["success", "reset", "normal"], parameters=["width", "offset", "tries"])
	gc.glitch_plot(plotdots={"success":"+g", "reset":"xr", "normal":None}, x_bound=(-48, 48), y_bound=(-48, 48))
	gc.set_global_step([8, 4, 2, 1])
	gc.set_range("tries",1,1)
	gc.set_step("tries",1)
	gc.set_range("width",width_min,width_max)
	gc.set_step("width",width_step)
	gc.set_range("offset",offset_min,offset_max)
	gc.set_step("offset",offset_step)
	
	for glitch_setting in gc.glitch_values():
		wb.glitch_controller_set(repeat = 1,offset = glitch_setting[1],ext_offset = 920, width = glitch_setting[0])
		wb.scope.arm()
		wb.write_simple_serial("g", bytearray([]))
		ret = wb.scope.capture()
		val = wb.read_simple_serial(4)
		if ret: # No response from target
			my_results.add("reset", (glitch_setting[0], glitch_setting[1], 1))
			wb.reset()
		else: # Response from target
			if (val['valid']==False) :
				my_results.add("reset" , (glitch_setting[0], glitch_setting[1], 1))
				pass
			else: # Communication ok
				gcnt = struct.unpack("<I", val['payload'])[0]
				expected=struct.unpack("<I", bytearray.fromhex("C4090000"))[0]
				if (gcnt == expected) : # Normal reponse
					my_results.add("normal" , (glitch_setting[0], glitch_setting[1], 1))
					pass
				else: #glitch!!!
					my_results.add("success", (glitch_setting[0], glitch_setting[1], 1))
					
	gc.calc()
	res=my_results.get_results()
	
	group_styles = {
		"success": {"color": "green", "marker": "o", "label": "Glitch Success"},
		"normal": {"color": "gray", "marker": "x", "label": "Normal"},
		"reset": {"color": "red", "marker": "s", "label": "Reset"},
	}

	plt.figure(figsize=(8, 6))

	for group, points in res.items():
		if points:
			width_vals, offset_vals, _ = zip(*points)
			plt.scatter(offset_vals, width_vals, c=group_styles[group]["color"],
						marker=group_styles[group]["marker"], label=group_styles[group]["label"])

	plt.xlabel("Offset")
	plt.ylabel("Width")
	plt.title("Glitch Sweep Results")
	plt.legend()
	plt.grid(True)
	plt.show()
	plt.show()

if __name__ == "__main__":
	main()


