#! /usr/bin/env python3

import chipwhisperer as cw 
import time 
import sys
import numpy as np
import argparse
import struct
from workbench import WorkBench
import readline


# --------------------------------------------------------------------------------------- #
#                                Glitch Attack parameters                                 #
# --------------------------------------------------------------------------------------- #

upload_firmware    = True
firmware           = "glitch_loop.hex"
default_ext_offset = 920
default_repeat     =   1
default_offset     = -12
default_width      =   8

# --------------------------------------------------------------------------------------- #
#                                                                                         #
# --------------------------------------------------------------------------------------- #

def main():
	"""Interact with the STM32 on the chipwhisperer board.
	"""
	usage = """Usage: %(prog)s [options]

%(prog)s is used to optionaly upload a firmware, and run some trace acquisition or timing measurement.

"""
	local_config = dict()
	
	wb = WorkBench(local_config.get('workbench', dict()), False)
	
	if upload_firmware:
		wb.upload_firmware(firmware)
		wb.reset()
		print("Firmware uploaded successfully")
	
	wb.reset()
	
	parser = argparse.ArgumentParser(usage=usage)
	parser.add_argument("action")
	parser.add_argument("-o", "--offset",
		default = default_offset,
		help = 'Glitch offset'
		)
	parser.add_argument("-w", "--width",
		default = default_width,
		help = 'Glitch width'
		)
	parser.add_argument("-r", "--repeat",
		default = default_repeat,
		help = 'Glitch repeat'
		)
	parser.add_argument("-e", "--ext_offset",
		default = default_ext_offset,
		help = 'Glitch external offset'
		)
		
	input_list = []
	while True:
		cmd = input(">")
		input_list.append(cmd)
		options = parser.parse_args(args=cmd.split(' '))
		
		if options.action == 'quit':
			break
		elif options.action == 'reset':
			print("Target Reset")
			wb.reset()
		elif options.action == 'normal':
			print("Normal execution")
			wb.write_simple_serial("g", bytearray([]))
			ret = wb.scope.capture()
			val = wb.read_simple_serial(4)
			if ret:
				print("Target Reset")
				wb.reset()
			else:
				if (val['valid']==False) :
					print("Target Reset")
				else:
					# gcnt is the loop counter
					gcnt = struct.unpack("<I", val['payload'])[0]
					expected=struct.unpack("<I", bytearray.fromhex("C4090000"))[0]
					if (gcnt == expected) :
						print("Target Normal Behavior")
					else: #glitch!!!
						print("Target ⚡ GLITCHED")
					print(gcnt)
					
		elif options.action == 'glitch':
			print("Performing glitch")
			print("parameters:" + " repeat: " + str(options.repeat) + ", offset : " + str(options.offset) + ", ext_offset: " + str(options.ext_offset) + ", width:" + str(options.width))
			wb.glitch_controller_set(float(options.repeat),float(options.offset),float(options.ext_offset),float(options.width))
			wb.scope.arm()
			# Send trigger
			wb.write_simple_serial("g", bytearray([]))
			ret = wb.scope.capture()
			val = wb.read_simple_serial(4)
			if ret:
				print("Target Reset")
				wb.reset()
			else:
				if (val['valid']==False) :
					print("Target Reset")
				else:
					# gcnt is the loop counter
					gcnt = struct.unpack("<I", val['payload'])[0]
					expected=struct.unpack("<I", bytearray.fromhex("C4090000"))[0]
					if (gcnt == expected) :
						print("Target Normal Behavior")
					else: #glitch!!!
						print("Target ⚡ GLITCHED")
					print(gcnt)
		else:
			print("Unknown command")

if __name__ == "__main__":
	main()


