#! /usr/bin/env python3

import chipwhisperer as cw 
import time 
import sys
import numpy as np
import argparse
import matplotlib.pyplot as plt
import struct 
#from tqdm import tqdm


class Device:

	def __init__(self, deviceID, target, friendlyName):
		self.deviceID = deviceID
		self.target = target
		self.friendlyName = friendlyName

KnownDevices = {
	# STM32F2 devices:
	Device(0x411, 'stm32f2', "STM32F20x and STM32F21x"),
}

class WorkBench:

	def __init__(self, cfg, verbose):
		self.verbose = verbose
		self.scope = cw.scope()
		self.scope.default_setup()
		if 'scope' in cfg:
			for k in cfg['scope']:
				if k == 'gain.db':
					offset = float(cfg['scope'][k])
					if self.verbose:
						print("[local] Offseting scope.{} by: {} dB".format(k, offset))
					self.scope.gain.db += offset

		self.target = cw.target(self.scope, cw.targets.SimpleSerial) ###self.target = cw.target(self.scope)
		# Glitch setup
		self.scope.adc.timeout=0.1
		self.scope.glitch.clk_src="clkgen"
		self.scope.glitch.output="clock_xor"  #glitch = clock xor(^) glitch
		self.scope.glitch.trigger_src="ext_single"   #glitch only a after scope.arm() called
		self.scope.trigger.triggers = "tio4"
		self.scope.io.hs2="glitch"  #output glitch_out on the clock line
		self.scope.glitch.repeat     = 1
		self.scope.glitch.ext_offset = 10
		self.scope.glitch.offset = 0
		self.scope.glitch.width = 1

	def upload_firmware(self, hex_file):
		if self.verbose:
			print("Uploading file '{}'".format(hex_file))

		prog = cw.programmers.STM32FProgrammer
		cw.program_target(self.scope, prog, hex_file)
	
	def write_simple_serial(self,cmd, data):
		self.target.simpleserial_write(cmd, data)
		
	def read_simple_serial(self,size):
		return self.target.simpleserial_read_witherrors('r' , size , glitch_timeout=10)  #use read OR read with errors

	def reset(self):
		# Reset the STM32 target.
		self.scope.io.nrst = 'low'
		time.sleep(0.020)
		self.scope.io.nrst = 'high_z'  ##instead of high
		time.sleep(0.050)
		
	def glitch_controller_set(self,repeat,offset,ext_offset,width):
		self.glitch_controller_set_repeat(repeat)
		self.glitch_controller_set_offset(offset)
		self.glitch_controller_set_ext_offset(ext_offset)
		self.glitch_controller_set_width(width)
		
	def glitch_controller_set_repeat(self,repeat):
		self.scope.glitch.repeat=repeat
		
	def glitch_controller_set_offset(self,offset):
		self.scope.glitch.offset = offset
		
	def glitch_controller_set_ext_offset(self,ext_offset):
		self.scope.glitch.ext_offset = ext_offset
		
	def glitch_controller_set_width(self,width):
		self.scope.glitch.width = width
